<?php

namespace App\Http\Controllers;

use App\Mail\SendInvoiceMail;
use App\Mail\SendOrderDetailsMail;
use App\Models\Category;
use App\Models\Testimonial;
use App\Models\Config;
use App\Models\LegalPage;
use App\Models\Order;
use App\Models\Post;
use App\Models\Product;
use App\Models\Question;
use App\Models\Server;
use App\Models\Service;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class AdminController extends Controller
{
    public function dashboard() {
        /**
         * LAST 14 DAYS;
        */
        $_14_days = collect(CarbonPeriod::since(Carbon::now()->subDays(13))->days(1)->until(Carbon::now()));
        foreach($_14_days as $day) {
            $startDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->startOfDay();
            $endDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->endOfDay();
            $orders = Order::whereBetween( 'created_at', [ $startDate, $endDate ] )->wherePaid(true)->whereRefunded(false)->get();
            $_14daysLabels[] = $day->format('d');
            $_14daysValues[] = $this->getSumRevenue($orders);
        }
        $_14days = [
            'labels' => $_14daysLabels,
            'values' => $_14daysValues,
        ];
        /**
         * LAST 30 DAYS;
        */
        $_30_days = collect(CarbonPeriod::since(Carbon::now()->subDays(30))->days(1)->until(Carbon::now()));
        foreach($_30_days as $day) {
            $startDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->startOfDay();
            $endDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->endOfDay();
            $orders = Order::whereBetween( 'created_at', [ $startDate, $endDate ] )->wherePaid(true)->whereRefunded(false)->get();
            $_30daysLabels[] = $day->format('d-m-Y');
            $_30daysValues[] = $this->getSumRevenue($orders);
        }
        $_30days = [
            'labels' => array_reverse($_30daysLabels),
            'values' => array_reverse($_30daysValues),
        ];
        /**
         * ORDER COUNTS;
        */
        $todayOrders = Order::whereDate('created_at', Carbon::today())->whereRefunded(false)->wherePaid(true);
        $yesterdayOrders = Order::whereDate('created_at', Carbon::yesterday())->whereRefunded(false)->wherePaid(true);
        $last7daysOrders = Order::where('created_at','>=', Carbon::now()->subdays(6))->whereRefunded(false)->wherePaid(true);
        $last30daysOrders = Order::where('created_at','>=', Carbon::now()->subdays(30))->whereRefunded(false)->wherePaid(true);
        $lifetimeOrders = Order::wherePaid(true)->whereRefunded(false)->get();
        $today = [
            'count' => $todayOrders->count(),
            'revenue' => $this->getSumRevenue($todayOrders->get()),
        ];
        $yesterday = [
            'count' => $yesterdayOrders->count(),
            'revenue' => $this->getSumRevenue($yesterdayOrders->get()),
        ];
        $last30days = [
            'count' => $last30daysOrders->count(),
            'revenue' => $this->getSumRevenue($last30daysOrders->get()),
        ];
        $last7days = [
            'count' => $last7daysOrders->count(),
            'revenue' => $this->getSumRevenue($last7daysOrders->get()),
        ];
        $lifetime = [
            'count' => $lifetimeOrders->count(),
            'revenue' => $this->getSumRevenue($lifetimeOrders),
        ];
        return view('admin.dashboard', compact('today', 'yesterday', 'last7days', 'last30days', 'lifetime', '_14days', '_30days'));
    }
    
    private function getSumRevenue($items) {
        $total = 0;
        foreach( $items as $item ) {
            $total = $total + floatval($item['total']);
        }
        return number_format($total, 2);
    }

    public function orders() {
        $title = 'Orders';
        $orders = tap(Order::wherePaid(true)->orderBy('payment_date', 'desc')->orderBy('payment_date', 'desc')->paginate(20))->map(function($order) {
            $order->products = $order->products;
            $order->date = $order->created_at->format('d-M-Y H:i');
            return $order;
        });
        return view('admin.orders')->with(compact('orders', 'title'));
    }

    public function invoices() {
        $title = 'Invoices';
        $orders = tap(Order::wherePaid(false)->orderBy('created_at', 'desc')->orderBy('created_at', 'desc')->paginate(20))->map(function($order) {
            $order->products = $order->products;
            $order->date = $order->created_at->format('d-M-Y H:i');
            return $order;
        });
        return view('admin.orders')->with(compact('orders', 'title'));
    }    
    
    public function order_delete($id) {
        $order = Order::findOrFail($id);
        $order->service()->delete();
        $order->delete();
        return redirect()->route('admin.orders');
    }

    public function invoice_delete($id) {
        $order = Order::findOrFail($id);
        $order->service()->delete();
        $order->delete();
        return redirect()->route('admin.invoices');
    }

    public function order($id) {
        $order = Order::findOrFail($id);
        return view('admin.order')->with('order', $order);
    }

    public function invoice($id) {
        $order = Order::findOrFail($id);
        return view('admin.invoice')->with('order', $order);
    }

    public function send_invoice_link(Request $request) {
        Mail::to($request->email)->send(new SendInvoiceMail($request->id));
        return back()->with('success', 'Email Sent Successfully');
    }

    public function set_refunded($id) {
        $order = Order::findOrFail($id);
        $order->update(['refunded' => !$order->refunded]);
        return redirect()->route('admin.order', $order->id)->with('success', 'Refund Status Updated');
    }
    
    public function set_paid($id) {
        $order = Order::findOrFail($id);
        $order->update(['paid' => !$order->paid]);
        return redirect()->route('admin.order', $order->id)->with('success', 'Payment Status Updated');
    }

    public function set_sent($id) {
        $order = Order::findOrFail($id);
        $order->update(['sent' => !$order->sent]);
        return redirect()->route('admin.order', $order->id)->with('success', 'Sending Status Updated');
    }

    public function send_order(Request $request) {
        Mail::to($request->email)->send(new SendOrderDetailsMail($request->id));
        $order = Order::findOrFail($request->id);
        $order->update(['sent' => true]);
        return back()->with('success', 'Email Sent Successfully');
    }

    public function order_update_service(Request $request)  {
        $data = $request->except(['id', '_token']);
        $crystal_server_id = Server::where('name', 'LIKE', '%crystal%')->first()->id;
        $dino_server_id = Server::where('name', 'LIKE', '%dino%')->first()->id;
        /**
         * M3U
         */
        if( $request->m3u == '' ) {
            // CRYSTAL;
            if( $crystal_server_id == $request->server_id && $request->username != '' && $request->password != '' ) {
                $data['m3u'] = $request->host.'/get.php?username='.$request->username.'&password='.$request->password.'&output=ts&type=m3u_plus';
            }
            // DINO;
            if( $dino_server_id == $request->server_id && $request->username != '' && $request->password != '' ) {
                $data['m3u'] = $request->host.'/get.php?username='.$request->username.'&password='.$request->password.'&type=m3u_plus&output=ts';
            }
        }
        /**
         * EPG
         */
        if( $request->epg == '' ) {
            // CRYSTAL;
            if( $crystal_server_id == $request->server_id && $request->username != '' && $request->password != '' ) {
                $data['epg'] = $request->host.'/xmltv.php?username='.$request->username.'&password='.$request->password;
            }
            // DINO;
            if( $dino_server_id == $request->server_id && $request->username != '' && $request->password != '' ) {
                $data['epg'] = $request->host.'/xmltv.php?username='.$request->username.'&password='.$request->password;
            }
        }
        /**
         * ENIGMA 2
         */
        if( $request->enigma == '' ) {
            // CRYSTAL;
            if( $crystal_server_id == $request->server_id && $request->username != '' && $request->password != '' ) {
                $data['enigma'] = 'wget -O /etc/enigma2/iptv.sh "'.$request->host.'/get.php?username='.$request->username.'&password='.$request->password.'&output=ts&type=enigma22_script" && chmod 777 /etc/enigma2/iptv.sh && /etc/enigma2/iptv.sh';
            }
            // DINO;
            if( $dino_server_id == $request->server_id && $request->username != '' && $request->password != '' ) {
                $data['enigma'] = 'wget -O /etc/enigma2/iptv.sh "'.$request->host.'/get.php?username='.$request->username.'&password='.$request->password.'&type=enigma22_script&output=ts" && chmod 777 /etc/enigma2/iptv.sh && /etc/enigma2/iptv.sh';
            }
        }
        if( $request->expiration_date !== null) {
            if( $request->activation_date == null ) {
                $activation_date = Carbon::now();
            }else {
                $activation_date = Carbon::createFromFormat('d F, Y', $request->activation_date);
            }
            $expiration_date = Carbon::createFromFormat('d F, Y', $request->expiration_date);
        if($activation_date->gt($expiration_date)) {
                return back()->with('error', 'Incorrect Date');
            }
        }else {
            $activation_date = Carbon::now();
            $expiration_date = Carbon::now()->addMonths($request->months_number);
        }
        $data['activation_date'] = $activation_date;
        $data['expiration_date'] = $expiration_date;
        Service::whereId($request->id)->first()->update($data);
        return back()->with('success', 'Service Updated');
    }

    /**
     * START SETTINGS
    */
    public function settings() {
        $configs = [];
        foreach (Config::all() as $config) {
            $configs[$config->name] = $config->value;
        }
        return view('admin.settings')->with('configs', $configs);
    }

    public function settings_save(Request $request) {
        $data = $request->all();
        if($request->hasFile('white_logo')) {
            $file = $request->file('white_logo');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $website_logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $website_logo);
            $data['white_logo'] = $website_logo;
        }
        if($request->hasFile('black_logo')) {
            $file = $request->file('black_logo');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $website_logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $website_logo);
            $data['black_logo'] = $website_logo;
        }
        if($request->hasFile('favicon')) {
            $file = $request->file('favicon');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $website_logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $website_logo);
            $data['favicon'] = $website_logo;
        }
        foreach ($data as $key => $value) { $this->UpdateConfig($key, $value); }
        session()->flash('success', '!!');
        return back();
    }

    private function UpdateConfig( $key, $value ) {
        Config::where('name', $key)->update(['value' => $value]);
    }
    /**
     * END SETTINGS
     * START PRODUCTS
    */
    public function products() {
        $products = Product::orderBy('created_at', 'desc')->get()->map(function($product) {
            $product->orders = $product->orders()->count();
            $product->poster = $product->poster == '' ? asset('noposter.jpeg') : asset('storage/'.$product->poster);
            return $product;
        });
        return view('admin.products')->with('products', $products);
    }

    public function product_insert() {
        $categories = Category::all();
        return view('admin.product_insert')->with(compact('categories'));
    }

    public function product_save(Request $request) {
        $newProductData = $request->all();
        if($request->hasFile('poster')) {
            $file = $request->file('poster');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $newProductData['poster'] = $poster;
        }else {
            $newProductData['poster'] = '';
        }
        $newProductData['slug'] = Str::slug($newProductData['title'], '-');
        Product::create($newProductData);
        return redirect()->route('admin.products')->with('success', 'Product Inserted!');
    }

    public function product_edit($id) {
        $product = Product::findOrFail($id);
        $categories = Category::all();
        return view('admin.product_edit')->with(compact('product', 'categories'));
    }

    public function product_delete($id) {
        Product::find($id)->delete();
        return redirect()->route('admin.products');
    }

    public function product_update(Request $request) {
        $data = $request->all();
        unset($data['_token']);
        if($request->hasFile('poster')) {
            $file = $request->file('poster');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $data['poster'] = $poster;
        }
        Product::find($request->id)->update($data);
        return redirect()->route('admin.product_edit', $request->id)->with('success', 'Product Updated!');
    }

    /**
     *  END PRODUCTS
     * START TESTIMONIALS
    */
    public function testimonials() {
        $testimonials = Testimonial::orderBy('created_at', 'desc')->get();
        return view('admin.testimonials')->with('testimonials', $testimonials);
    }

    public function testimonial_insert() {
        return view('admin.testimonial_insert');
    }


    public function testimonial_edit($id) {
        $testimonial = Testimonial::findOrFail($id);
        return view('admin.testimonial_edit')->with('testimonial', $testimonial);
    }

    public function testimonial_delete($id) {
        Testimonial::find($id)->delete();
        return redirect()->route('admin.testimonials');
    }

    public function testimonial_save(Request $request) {
        $newProductData = $request->all();
        unset($newProductData['_token']);
        if($request->hasFile('poster')) {
            $file = $request->file('poster');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $newProductData['poster'] = $poster;
        }else {
            $newProductData['poster'] = '';
        }
        Testimonial::create($newProductData);
        return redirect()->route('admin.testimonials')->with('success', 'Testimonial Inserted!');
    }

    public function testimonial_update(Request $request) {
        $data = $request->all();
        unset($data['_token']);
        if($request->hasFile('poster')) {
            $file = $request->file('poster');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $data['poster'] = $poster;
        }
        Testimonial::find($request->id)->update($data);
        return redirect()->route('admin.testimonial_edit', $request->id)->with('success', 'Testimonial Updated!');
    }

    /**
     * END TESTIMONIALS
     * START CATEGORIES
    */
    public function categories() {
        $categories = Category::all();
        return view('admin.categories')->with('categories', $categories);
    }

    public function category_insert() {
        return view('admin.category_insert');
    }

    public function category_save(Request $request) {
        Category::updateOrCreate(['name' => $request->name, 'seo_description' => $request->seo_description]);
        return redirect()->route('admin.categories');
    }

    public function category_delete($id) {
        Category::find($id)->delete();
        return redirect()->route('admin.categories');
    }

    public function categories_update(Request $request) {
        $ids = $request->id;
        $names = $request->name;
        $seo_descriptions = $request->seo_description;
        for( $i = 0; $i < count($ids); $i ++ ) {
            Category::find($ids[$i])->update( ['name' => $names[$i], 'seo_description' => $seo_descriptions[$i] ] );
        }
        return redirect()->route('admin.categories');
    }
    /**
     * END CATEGORIES
     * START FAQ
    */
    public function questions() {
        $questions = Question::all();
        return view('admin.questions')->with('questions', $questions);
    }

    public function questions_insert() {
        return view('admin.questions_insert');
    }

    public function questions_edit($id) {
        $question = Question::whereId($id)->first();
        return view('admin.questions_edit')->with('question', $question);
    }

    public function questions_save(Request $request) {
        Question::create([
            'question' => $request->question,
            'answer' => $request->answer,
            'visibility' => 1,
        ]);
        return back()->with('success', true);
    }

    public function questions_update(Request $request) {
        Question::whereId($request->id)->first()->update([
            'question' => $request->question,
            'answer' => $request->answer,
            'visibility' => $request->visibility,
        ]);
        return back()->with('success', true);
    }

    public function questions_delete($id) {
        Question::find($id)->delete();
        return redirect()->route('admin.questions');
    }
    /**
     * END FAQ
     * START POSTS
    */
    public function posts() {
        $posts = Post::orderBy('created_at', 'desc')->get();
        return view('admin.posts')->with('posts', $posts);
    }

    public function post_insert() {
        return view('admin.post_insert');
    }

    public function post_save(Request $request) {
        $newPost = new Post();
        if($request->hasFile('poster')) {
            $file = $request->file('poster');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $newPost->poster = $poster;
        }else $newPost->poster = '';
        $newPost->title = $request->title;
        $newPost->slug = Str::slug($newPost->title, '-');
        $newPost->visibility = $request->visibility;
        $newPost->content = $request->content;
        $newPost->seo_title = $request->seo_title;
        $newPost->seo_keywords = $request->seo_keywords;
        $newPost->seo_description = $request->seo_description;
        $newPost->save();
        return redirect()->route('admin.posts');
    }

    public function post_delete($id) {
        Post::find($id)->delete();
        return redirect()->route('admin.posts');
    }

    public function post_edit($id)
    {
        $post = Post::find($id);
        return view('admin.post_edit')->with('post', $post);
    }

    public function post_update(Request $request) {
        $data = [];
        $data['title'] = $request->title;
        if($request->hasFile('poster')) {
            $file = $request->file('poster');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $data['poster'] = $poster;
        }
        $data['visibility'] = (int) $request->visibility;
        $data['slug'] = $request->slug;
        $data['content'] = $request->content;
        $data['seo_title'] = $request->seo_title;
        $data['seo_keywords'] = $request->seo_keywords;
        $data['seo_description'] = $request->seo_description;
        Post::whereId($request->id)->first()->update($data);
        return back()->with('success', true);
    }

    /**
     * LEGAL PAGES
    */
    public function legal_pages() {
        $pages = LegalPage::all();
        return view('admin.legal_pages')->with('pages', $pages);
    }

    public function legal_pages_insert() {
        return view('admin.legal_pages_insert');
    }

    public function legal_pages_save(Request $request) {
        $page = new LegalPage();
        $page->slug = Str::slug($request->title, '-');
        $page->title = $request->title;
        $page->visibility = $request->visibility;
        $page->content = $request->content;
        $page->save();
        return redirect()->route('admin.legal_pages');
    }

    public function legal_pages_delete($id) {
        LegalPage::find($id)->delete();
        return redirect()->route('admin.legal_pages');
    }

    public function legal_pages_edit($id)
    {
        $page = LegalPage::find($id);
        return view('admin.legal_pages_edit')->with('page', $page);
    }

    public function legal_pages_update(Request $request)
    {
        LegalPage::whereId($request->id)->first()->update([
            'title' => $request->title,
            'slug' => $request->slug,
            'visibility' => $request->visibility,
            'content' => $request->content
        ]);
        return back()->with('success', true);
    }

}