<?php

namespace App\Http\Controllers;

use App\Mail\SendInvoiceMail;
use App\Mail\SendOrderDetailsMail;
use App\Models\Category;
use App\Models\Cnumber;
use App\Models\Testimonial;
use App\Models\Config;
use App\Models\Connection;
use App\Models\Currency;
use App\Models\LegalPage;
use App\Models\Order;
use App\Models\Paypal;
use App\Models\Post;
use App\Models\Product;
use App\Models\Question;
use App\Models\Server;
use App\Models\Subtype;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class AdminController extends Controller
{
    public function dashboard() {
        $currency_id = Config::whereName('currency')->first()->value;
        $currency = Currency::whereId($currency_id)->first();
        /**
         * LAST 14 DAYS;
        */
        $_14_days = collect(CarbonPeriod::since(Carbon::now()->subDays(13))->days(1)->until(Carbon::now()));
        foreach($_14_days as $day) {
            $startDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->startOfDay();
            $endDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->endOfDay();
            $orders = Order::whereBetween( 'created_at', [ $startDate, $endDate ] )->wherePaid(true)->whereRefunded(false)->get();
            $_14daysLabels[] = $day->format('d');
            $_14daysValues[] = $this->getSumRevenue($orders);
        }
        $_14days = [
            'labels' => $_14daysLabels,
            'values' => $_14daysValues,
        ];
        /**
         * LAST 30 DAYS;
        */
        $_30_days = collect(CarbonPeriod::since(Carbon::now()->subDays(30))->days(1)->until(Carbon::now()));
        foreach($_30_days as $day) {
            $startDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->startOfDay();
            $endDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->endOfDay();
            $orders = Order::whereBetween( 'payment_date', [ $startDate, $endDate ] )->wherePaid(true)->whereRefunded(false)->get();
            $_30daysLabels[] = $day->format('d-m-Y');
            $_30daysValues[] = $this->getSumRevenue($orders);
        }
        $_30days = [
            'labels' => array_reverse($_30daysLabels),
            'values' => array_reverse($_30daysValues),
        ];
        /**
         * ORDER COUNTS;
        */
        $todayOrders = Order::whereDate('payment_date', Carbon::today())->whereRefunded(false)->wherePaid(true);
        $yesterdayOrders = Order::whereDate('payment_date', Carbon::yesterday())->whereRefunded(false)->wherePaid(true);
        $last7daysOrders = Order::where('payment_date','>=', Carbon::now()->subdays(6))->whereRefunded(false)->wherePaid(true);
        $last30daysOrders = Order::where('payment_date','>=', Carbon::now()->subdays(30))->whereRefunded(false)->wherePaid(true);
        $lifetimeOrders = Order::wherePaid(true)->whereRefunded(false)->get();
        $today = [
            'count' => $todayOrders->count(),
            'revenue' => $this->getSumRevenue($todayOrders->get()),
        ];
        $yesterday = [
            'count' => $yesterdayOrders->count(),
            'revenue' => $this->getSumRevenue($yesterdayOrders->get()),
        ];
        $last30days = [
            'count' => $last30daysOrders->count(),
            'revenue' => $this->getSumRevenue($last30daysOrders->get()),
        ];
        $last7days = [
            'count' => $last7daysOrders->count(),
            'revenue' => $this->getSumRevenue($last7daysOrders->get()),
        ];
        $lifetime = [
            'count' => $lifetimeOrders->count(),
            'revenue' => $this->getSumRevenue($lifetimeOrders),
        ];
        return view('admin.dashboard', compact(
            'today', 'yesterday', 'last7days', 'last30days', 'lifetime', '_14days', '_30days', 'currency'
        ));
    }
    
    private function getSumRevenue($items) {
        $total = 0;
        foreach( $items as $item ) {
            $total = $total + floatval($item['total']);
        }
        return number_format($total, 2);
    }

    public function orders(Request $request) {
        $title = 'Orders';
        $currency_id = Config::whereName('currency')->first()->value;
        $currency = Currency::whereId($currency_id)->first();
        if( $request->fastdelivery == 1 ) {
            $ordersDb = Order::wherePaid(true)
                                ->whereSent(false)
                                ->whereFastdelivery(true)
                                ->orderBy('payment_date', 'desc')
                                ->orderBy('payment_date', 'desc')
                                ->paginate(20);
        }else {
            $ordersDb = Order::wherePaid(true)
                                ->orderBy('payment_date', 'desc')
                                ->orderBy('payment_date', 'desc')
                                ->paginate(20);
        }
        $orders = tap($ordersDb)->map(function($order) {
            $order->products = $order->products;
            $order->date = $order->payment_date->format('d-M-Y H:i');
            return $order;
        });
        return view('admin.orders')->with(compact('orders', 'title', 'currency'));
    }

    public function orders_filter(Request $request) {
        $title = 'Orders';
        $currency_id = Config::whereName('currency')->first()->value;
        $currency = Currency::whereId($currency_id)->first();
        $search_type = $request->search_type;
        $search_input = $request->search_input;
        // Orders DB;
        $ordersDb = Order::wherePaid(true)
                        ->where($search_type, $search_input)
                        ->orderBy('payment_date', 'desc')
                        ->orderBy('payment_date', 'desc')
                        ->paginate(20);
        // Collect Other Data;
        $orders = tap($ordersDb)->map(function($order) {
            $order->products = $order->products;
            $order->date = $order->payment_date->format('d-M-Y H:i');
            return $order;
        });
        return view('admin.orders')->with(compact('orders', 'title', 'currency', 'search_type', 'search_input'));
    }
    
    public function order_delete($id) {
        $order = Order::findOrFail($id);
        $paid = $order->paid;
        $order->connections()->delete();
        $order->connections()->detach();
        $order->delete();
        return redirect()->route( $paid ? 'admin.orders' : 'admin.invoices');
    }

    public function order($id) {
        $order = Order::findOrFail($id);
        $default_iptvServer = (int) Config::whereName('default_iptvServer')->first()->value;
        $label = '#'.$order->id.' '.explode('@', $order->billing_email)[0];
        return view('admin.order')->with(compact('order', 'default_iptvServer', 'label'));
    }

    public function invoices() {
        $title = "Invoices";
        $currency_id = Config::whereName('currency')->first()->value;
        $currency = Currency::whereId($currency_id)->first();
        $orders = tap(Order::wherePaid(false)->orderBy('created_at', 'desc')->orderBy('created_at', 'desc')->paginate(20))->map(function($order) {
            $order->products = $order->products;
            $order->date = $order->created_at->format('d-M-Y H:i');
            return $order;
        });
        return view('admin.orders')->with(compact('orders', 'title', 'currency'));
    } 

    public function invoice($id) {
        $invoice = Order::findOrFail($id);
        if($invoice->paid) return redirect()->route('admin.order', $invoice->id);
        return view('admin.invoice')->with('order', $invoice);
    }

    public function send_invoice_link(Request $request) {
        Mail::to($request->email)->send(new SendInvoiceMail($request->id));
        return back()->with('success', 'Email Sent Successfully');
    }

    public function set_refunded($id) {
        $order = Order::findOrFail($id);
        $order->update(['refunded' => !$order->refunded]);
        return redirect()->route('admin.order', $order->id)->with('success', 'Refund Status Updated');
    }
    
    public function set_paid($id) {
        $order = Order::findOrFail($id);
        $order->update(['paid' => !$order->paid]);
        return redirect()->route('admin.order', $order->id)->with('success', 'Payment Status Updated');
    }

    public function set_sent($id) {
        $order = Order::findOrFail($id);
        $order->update(['sent' => !$order->sent]);
        return redirect()->route('admin.order', $order->id)->with('success', 'Sending Status Updated');
    }

    public function send_order(Request $request) {
        $order = Order::findOrFail($request->id);
        Mail::to($request->email)->send(new SendOrderDetailsMail($request->id));
        $order = Order::findOrFail($request->id);
        $order->update(['sent' => true]);
        return back()->with('success', 'Email Sent Successfully');
    }

    public function update_order(Request $request) {
        Order::whereId($request->id)->first()->update($request->except(['_token', 'id']));
        return back()->with('success', 'Order Updated Successfully');
    }

    public function update_connection(Request $request)  {
        $data = $request->except(['id', '_token']);
        $host = $request->host;
        // LIST SERVERS;
        $crystal = Server::where('name', 'LIKE', '%crystal%')->first();
        $dino = Server::where('name', 'LIKE', '%dino%')->first();
        $trex = Server::where('name', 'LIKE', '%trex%')->first();
        // Selected Server;
        $selected_server = $request->server_id;
        // Check;
        if( $selected_server == $crystal->id || $selected_server == $dino->id || $selected_server == $trex->id  ) {
            /**
             * GET Subscription Type
            */
            $subtype = Subtype::whereId($request->subtype_id)->first()->title;
            /**
             * MAG CHECK;
            */
            $is_mag = str_contains(strtolower($subtype), 'portal');


            if( $is_mag ) {
                /**
                 * GET HOST
                */
                if( $host == null ) {
                    if( $selected_server == $crystal->id ) $host = $crystal->default_mag;
                    if( $selected_server == $dino->id ) $host = $dino->default_mag;
                    if( $selected_server == $trex->id ) $host = $trex->default_mag;
                }
                /**
                 * UPDATE HOST;
                */
                $data['username'] = '';
                $data['password'] = '';
                $data['m3u'] = '';
                $data['epg'] = '';
                $data['enigma'] = '';
            }else {
                /**
                 * GET HOST
                */
                if( $host == null ) {
                    if( $selected_server == $crystal->id ) $host = $crystal->default_line;
                    if( $selected_server == $dino->id ) $host = $dino->default_line;
                    if( $selected_server == $trex->id ) $host = $trex->default_line;
                }
                /**
                 * GENERATE CONNECTION DETAILS;
                */
                if( $request->username !== null && $request->password !== null ) {
                    /**
                     * GENERATE M3U;
                    */
                    $data['m3u'] = $host.'/get.php?username='.$request->username.'&password='.$request->password.'&output=ts&type=m3u_plus';
                    /**
                     * GENERATE EPG;
                    */
                    $data['epg'] = $host.'/xmltv.php?username='.$request->username.'&password='.$request->password;
                    /**
                     * GENERATE ENIGMA;
                    */
                    $data['enigma'] = 'wget -O /etc/enigma2/iptv.sh "'.$host.'/get.php?username='.$request->username.'&password='.$request->password.'&output=ts&type=enigma22_script" && chmod 777 /etc/enigma2/iptv.sh && /etc/enigma2/iptv.sh';
                }
            }
        }
        $data['host'] = $host;
        if( $request->expiration_date !== null) {
            if( $request->activation_date == null ) {
                $activation_date = Carbon::now();
            }else {
                $activation_date = Carbon::createFromFormat('d F, Y', $request->activation_date);
            }
            $expiration_date = Carbon::createFromFormat('d F, Y', $request->expiration_date);
        if($activation_date->gt($expiration_date)) {
                return back()->with('error', 'Incorrect Date');
            }
        }else {
            $activation_date = Carbon::now();
            $expiration_date = Carbon::now()->addMonths($request->months_number);
        }
        $data['activation_date'] = $activation_date;
        $data['expiration_date'] = $expiration_date;
        
        Connection::whereId($request->id)->first()->update($data);
        
        $connection = Connection::whereId($request->id)->first();

        return $connection;
        return back()->with('selected_connection_id', $request->id);
    }

    /**
     * START SETTINGS
    */
    public function settings() {
        $configs = [];
        foreach (Config::all() as $config) {
            $configs[$config->name] = $config->value;
        }
        return view('admin.settings')->with('configs', $configs);
    }

    public function settings_save(Request $request) {
        $data = $request->all();
        if($request->hasFile('white_logo')) {
            $file = $request->file('white_logo');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $website_logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $website_logo);
            $data['white_logo'] = $website_logo;
        }
        if($request->hasFile('black_logo')) {
            $file = $request->file('black_logo');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $website_logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $website_logo);
            $data['black_logo'] = $website_logo;
        }
        if($request->hasFile('favicon')) {
            $file = $request->file('favicon');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $website_logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $website_logo);
            $data['favicon'] = $website_logo;
        }
        foreach ($data as $key => $value) { $this->UpdateConfig($key, $value); }
        session()->flash('success', '!!');
        return back();
    }

    public function homepage_settings() {
        $configs = [];
        foreach (Config::where('name', 'like', 'homepage_%')->get() as $config) {
            $configs[$config->name] = $config->value;
        }
        return view('admin.homepage_settings')->with('configs', $configs);
    }

    private function UpdateConfig( $key, $value ) {
        Config::where('name', $key)->update(['value' => $value]);
    }
    /**
     * END SETTINGS
     * START PAYPAL
    */

    public function paypal() {
        $is_shuffled = Config::whereName('paypal_shuffled')->first()->value;
        $paypals = Paypal::all();
        return view('admin.paypal')->with(compact('paypals', 'is_shuffled'));
    }

    public function paypal_insert() {
        return view('admin.paypal_insert');
    }

    public function paypal_save(Request $request) {
        Paypal::create( $request->only( 'email', 'client_id', 'active' ) );
        return redirect()->route('admin.paypal')->with('success', 'PayPal Inserted');
    }

    public function paypal_delete($id) {
        Paypal::whereId($id )->delete();
        return redirect()->route('admin.paypal')->with('success', 'PayPal Removed');
    }

    public function paypal_update(Request $request) {
        $ids = $request->ids;
        $emails = $request->emails;
        $client_ids = $request->client_ids;
        for($i = 0; $i < count($emails); $i ++) {
            Paypal::whereId($ids[$i])->update([
                'email' => $emails[$i],
                'client_id' => $client_ids[$i],
            ]);
        }
        return back();
    }

    public function paypal_shuffled($current_status) {
        switch($current_status) {
            case 0:
                $new_status = 1;
                break;
            case 1:
                $new_status = 0;
                break;
        }
        Config::whereName('paypal_shuffled')->update([ 'value' =>  $new_status]);
        return back();
    }

    public function paypal_status($id) {
        $current_status = Paypal::whereId($id)->first()->active;
        switch($current_status) {
            case 0:
                $new_status = 1;
                break;
            case 1:
                $new_status = 0;
                break;
        }
        Paypal::whereId($id)->update([ 'active' =>  $new_status]);
        return back();
    }

    /**
     * END PAYPAL
     * START PRODUCTS
    */
    public function products() {
        $currency_id = Config::whereName('currency')->first()->value;
        $currency = Currency::whereId($currency_id)->first();
        $products = Product::orderBy('created_at', 'desc')->get()->map(function($product) {
            $product->orders = $product->orders()->count();
            return $product;
        });
        return view('admin.products')->with(compact('products', 'currency'));
    }

    public function product_insert() {
        $categories = Category::all();
        return view('admin.product_insert')->with(compact('categories'));
    }

    public function product_save(Request $request) {
        $newProductData = $request->all();
        if($request->hasFile('poster')) {
            $file = $request->file('poster');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $newProductData['poster'] = $poster;
        }else {
            $newProductData['poster'] = '';
        }
        $newProductData['slug'] = Str::slug($newProductData['title'], '-');
        Product::create($newProductData);
        return redirect()->route('admin.products')->with('success', 'Product Inserted!');
    }

    public function product_edit($id) {
        $product = Product::findOrFail($id);
        $categories = Category::all();
        return view('admin.product_edit')->with(compact('product', 'categories'));
    }

    public function product_delete($id) {
        Product::find($id)->delete();
        return redirect()->route('admin.products');
    }

    public function product_update(Request $request) {
        $data = $request->all();
        unset($data['_token']);
        unset($data['id']);
        if($request->hasFile('poster')) {
            $file = $request->file('poster');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $data['poster'] = $poster;
        }
        Product::find($request->id)->update($data);
        return redirect()->route('admin.product_edit', $request->id)->with('success', 'Product Updated!');
    }
    
    /**
     * END PRODUCTS
     * START SUBTYPES
    */
    public function subtypes() {
        $subtypes = Subtype::all();
        return view('admin.subtypes')->with('subtypes', $subtypes);
    }

    public function subtypes_update(Request $request) {
        $ids = $request->id;
        $titles = $request->title;
        for( $i = 0; $i < count($ids); $i ++ ) {
            Subtype::find($ids[$i])->update( [ 'title' => $titles[$i] ] );
        }
        return redirect()->route('admin.subtypes')->with('success', 'Subtypes Updated Successfully!');
    }

    public function subtype_insert () {
        return view('admin.subtype_insert');
    }
    
    public function subtype_save(Request $request) {
        Subtype::create($request->except(['token']));
        return redirect()->route('admin.subtypes')->with('success', 'Subtype Inserted Successfully!');
    }

    /**
     * END SUBTYPES
     * START SERVERS
    */
    public function servers() {
        $servers = Server::all();
        return view('admin.servers')->with('servers', $servers);
    }

    public function server_insert () {
        return view('admin.server_insert');
    }

    public function server_delete($id) {
        $server = Server::findOrFail($id);
        $server->delete();
        return redirect()->route( 'admin.servers');
    }    

    public function servers_update(Request $request) {
        $ids = $request->id;
        $names = $request->name;
        $lines = $request->line;
        $mags = $request->mag;
        for( $i = 0; $i < count($ids); $i ++ ) {
            Server::find($ids[$i])->update([
                'name' => $names[$i],
                'default_line' => $lines[$i],
                'default_mag' => $mags[$i]
            ]);
        }
        return redirect()->route('admin.servers')->with('success', 'Servers Updated Successfully!');
    }

    public function server_save(Request $request) {
        Server::create($request->except(['token']));
        return redirect()->route('admin.servers')->with('success', 'Server Inserted Successfully!');
    }
    /**
     * END SERVERS
     * START CATEGORIES
    */
    public function categories() {
        $categories = Category::all();
        return view('admin.categories')->with('categories', $categories);
    }

    public function category_insert() {
        return view('admin.category_insert');
    }

    public function category_save(Request $request) {
        Category::updateOrCreate(['name' => $request->name, 'active' => $request->active]);
        return redirect()->route('admin.categories');
    }

    public function category_delete($id) {
        Category::find($id)->delete();
        return redirect()->route('admin.categories');
    }

    public function categories_update(Request $request) {
        $ids = $request->id;
        $names = $request->name;
        $actives = $request->active;
        for( $i = 0; $i < count($ids); $i ++ ) {
            Category::find($ids[$i])->update( ['name' => $names[$i], 'active' => $actives[$i] ] );
        }
        return redirect()->route('admin.categories');
    }
    /**
     * END CATEGORIES
     * START CNUMBERS
     * START FAQ
    */
    public function cnumbers() {
        $cnumbers = Cnumber::all();
        $categories = Category::all();
        return view('admin.cnumbers')->with(compact('cnumbers', 'categories'));
    }

    public function cnumber_insert() {
        return view('admin.cnumber_insert')->with('categories', Category::all());
    }

    public function cnumber_save(Request $request) {
        Cnumber::updateOrCreate([
            'name' => $request->name,
            'category_id' => $request->category_id,
            'number' => $request->number,
            'active' => $request->active
        ]);
        return redirect()->route('admin.cnumbers');
    }

    public function cnumber_delete($id) {
        Cnumber::find($id)->delete();
        return redirect()->route('admin.cnumbers');
    }

    public function cnumbers_update(Request $request) {
        $ids = $request->id;
        $names = $request->name;
        $numbers = $request->number;
        $categories = $request->category;
        $actives = $request->active;
        for( $i = 0; $i < count($ids); $i ++ ) {
            Cnumber::find($ids[$i])->update([
                'name' => $names[$i],
                'category_id' => $categories[$i],
                'number' => $numbers[$i],
                'active' => $actives[$i] 
            ]);
        }
        return redirect()->route('admin.cnumbers')->with('success', 'Cnumbers Updated Successfully!!');
    }
    /**
     * END CNUMBERS
     * START FAQ
    */
    public function questions() {
        $questions = Question::all();
        return view('admin.questions')->with('questions', $questions);
    }

    public function questions_insert() {
        return view('admin.questions_insert');
    }

    public function questions_edit($id) {
        $question = Question::whereId($id)->first();
        return view('admin.questions_edit')->with('question', $question);
    }

    public function questions_save(Request $request) {
        Question::create([
            'question' => $request->question,
            'answer' => $request->answer,
            'visibility' => 1,
        ]);
        return redirect(route('admin.questions'))->with('success', "Question Inserted Successfully.");
    }

    public function questions_update(Request $request) {
        Question::whereId($request->id)->first()->update([
            'question' => $request->question,
            'answer' => $request->answer,
            'visibility' => $request->visibility,
        ]);
        return back()->with('success', true);
    }

    public function questions_delete($id) {
        Question::find($id)->delete();
        return redirect()->route('admin.questions');
    }
    /**
     * END FAQ
     * START POSTS
    */
    public function posts() {
        $posts = Post::orderBy('created_at', 'desc')->get();
        return view('admin.posts')->with('posts', $posts);
    }

    public function post_insert() {
        return view('admin.post_insert');
    }

    public function post_save(Request $request) {
        $newPost = new Post();
        if($request->hasFile('icon')) {
            $file = $request->file('icon');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $icon = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $icon);
            $newPost->icon = $icon;
        }else $newPost->icon = '';
        $newPost->title = $request->title;
        $newPost->slug = Str::slug($newPost->title, '-');
        $newPost->visibility = $request->visibility;
        $newPost->content = $request->content;
        $newPost->save();
        return redirect()->route('admin.posts');
    }

    public function post_delete($id) {
        Post::find($id)->delete();
        return redirect()->route('admin.posts');
    }

    public function post_edit($id) {
        $post = Post::find($id);
        return view('admin.post_edit')->with('post', $post);
    }

    public function post_update(Request $request) {
        $data = [];
        if($request->hasFile('icon')) {
            $file = $request->file('icon');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $icon = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $icon);
            $data['icon'] = $icon;
        }
        $data['title'] = $request->title;
        $data['visibility'] = (int) $request->visibility;
        $data['slug'] = $request->slug;
        $data['content'] = $request->content;
        Post::whereId($request->id)->first()->update($data);
        return back()->with('success', true);
    }
    /**
     * LEGAL PAGES
    */
    public function legal_pages() {
        $pages = LegalPage::all();
        return view('admin.legal_pages')->with('pages', $pages);
    }

    public function legal_pages_insert() {
        return view('admin.legal_pages_insert');
    }

    public function legal_pages_save(Request $request) {
        $page = new LegalPage();
        $page->slug = Str::slug($request->title, '-');
        $page->title = $request->title;
        $page->visibility = $request->visibility;
        $page->content = $request->content;
        $page->save();
        return redirect()->route('admin.legal_pages');
    }

    public function legal_pages_delete($id) {
        LegalPage::find($id)->delete();
        return redirect()->route('admin.legal_pages');
    }

    public function legal_pages_edit($id)
    {
        $page = LegalPage::find($id);
        return view('admin.legal_pages_edit')->with('page', $page);
    }

    public function legal_pages_update(Request $request)
    {
        LegalPage::whereId($request->id)->first()->update([
            'title' => $request->title,
            'slug' => $request->slug,
            'visibility' => $request->visibility,
            'content' => $request->content
        ]);
        return back()->with('success', true);
    }


    /**
     * MEDIA
    */
    public function media() {
        $disk = Storage::disk('public');
        $files = $disk->allFiles();
        $fileData = collect();
        foreach($files as $file) {
            $fileData->push([
                'file' => $file,
                'date' => $disk->lastModified( $file )
            ]);
        }
        $files = $fileData->sortByDesc('date')->values()->all();
        return view('admin.media')->with('files', $files);
    }

    public function media_insert() {
        return view('admin.media_insert');
    }

    public function media_save(Request $request) {
        if($request->hasFile('myFile')) {
            $file = $request->file('myFile');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $icon = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $icon);
        }
        return redirect()->route('admin.media');
    }

}
