<?php

namespace App\Http\Controllers;

use App\Mail\SendOrderDetailsMail;
use App\Models\Category;
use App\Models\Config;
use App\Models\LegalPage;
use App\Models\Order;
use App\Models\Origin;
use App\Models\Paypal;
use App\Models\Product;
use App\Models\Question;
use App\Models\Service;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class AdminController extends Controller
{
    public function dashboard() {
        /**
         * LAST 14 DAYS;
        */
        $_14_days = collect(CarbonPeriod::since(Carbon::now()->subDays(13))->days(1)->until(Carbon::now()));
        foreach($_14_days as $day) {
            $startDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->startOfDay();
            $endDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->endOfDay();
            $orders = Order::whereBetween( 'created_at', [ $startDate, $endDate ] )->get();
            $_14daysLabels[] = $day->format('d');
            $_14daysValues[] = $this->getSumRevenue($orders);
        }
        $_14days = [
            'labels' => $_14daysLabels,
            'values' => $_14daysValues,
        ];
        /**
         * LAST 30 DAYS;
        */
        $_30_days = collect(CarbonPeriod::since(Carbon::now()->subDays(30))->days(1)->until(Carbon::now()));
        foreach($_30_days as $day) {
            $startDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->startOfDay();
            $endDate = Carbon::createFromFormat('Y-m-d', $day->toDateString())->endOfDay();
            $orders = Order::whereBetween( 'created_at', [ $startDate, $endDate ] )->get();
            $_30daysLabels[] = $day->format('d-m-Y');
            $_30daysValues[] = $this->getSumRevenue($orders);
        }
        $_30days = [
            'labels' => array_reverse($_30daysLabels),
            'values' => array_reverse($_30daysValues),
        ];
        /**
         * ORDER COUNTS;
        */
        $todayOrders = Order::whereDate('created_at', Carbon::today());
        $yesterdayOrders = Order::whereDate('created_at', Carbon::yesterday());
        $last7daysOrders = Order::where('created_at','>=', Carbon::now()->subdays(6));
        $last30daysOrders = Order::where('created_at','>=', Carbon::now()->subdays(30));
        $lifetimeOrders = Order::get();
        $today = [
            'count' => $todayOrders->count(),
            'revenue' => $this->getSumRevenue($todayOrders->get()),
        ];
        $yesterday = [
            'count' => $yesterdayOrders->count(),
            'revenue' => $this->getSumRevenue($yesterdayOrders->get()),
        ];
        $last30days = [
            'count' => $last30daysOrders->count(),
            'revenue' => $this->getSumRevenue($last30daysOrders->get()),
        ];
        $last7days = [
            'count' => $last7daysOrders->count(),
            'revenue' => $this->getSumRevenue($last7daysOrders->get()),
        ];
        $lifetime = [
            'count' => $lifetimeOrders->count(),
            'revenue' => $this->getSumRevenue($lifetimeOrders),
        ];
        return view('admin.dashboard', compact('today', 'yesterday', 'last7days', 'last30days', 'lifetime', '_14days', '_30days'));
    }

    private function getSumRevenue($items) {
        $total = 0;
        foreach( $items as $item ) {
            $total = $total + floatval($item['total']);
        }
        return $total;
    }

    public function orders() {
        $orders = tap(Order::orderBy('created_at', 'desc')->orderBy('created_at', 'desc')->paginate(2))->map(function($order) {
            $order->products = $order->products;
            $order->date = $order->created_at->format('d-M-Y H:i');
            return $order;
        });
        return view('admin.orders')->with('orders', $orders);
    }

    
    public function order_delete($id) {
        $order = Order::findOrFail($id);
        $order->service()->delete();
        $order->delete();
        return redirect()->route('admin.orders');
    }

    public function order($id) {
        $order = Order::findOrFail($id);
        return view('admin.order')->with('order', $order);
    }

    public function set_refunded($id) {
        $order = Order::findOrFail($id);
        $order->update(['refunded' => !$order->refunded]);
        return redirect()->route('admin.order', $order->id)->with('success', 'Refund Status Updated');
    }

    public function set_paid($id) {
        $order = Order::findOrFail($id);
        $order->update(['paid' => !$order->paid]);
        return redirect()->route('admin.order', $order->id)->with('success', 'Payment Status Updated');
    }

    public function set_sent($id) {
        $order = Order::findOrFail($id);
        $order->update(['sent' => !$order->sent]);
        return redirect()->route('admin.order', $order->id)->with('success', 'Sending Status Updated');
    }

    public function send_order(Request $request) {
        Mail::to($request->email)->send(new SendOrderDetailsMail($request->id));
        return back()->with('success', 'Email Sent Successfully');
    }

    public function order_update_service(Request $request)  {
        $data = $request->except(['id', '_token']);
        if( $request->activation_date !== null && $request->expiration_date !== null) {
            $activation_date = Carbon::createFromFormat('d F, Y', $request->activation_date);
            $expiration_date = Carbon::createFromFormat('d F, Y', $request->expiration_date);
            if($activation_date->gt($expiration_date)) {
                return back()->with('error', 'Incorrect Date');
            }
            $data['activation_date'] = $activation_date;
            $data['expiration_date'] = $expiration_date;
        }
        Service::whereId($request->id)->first()->update($data);
        return back()->with('success', 'Service Updated');
    }

    /**
     * START SETTINGS
    */
    public function settings() {
        $configs = [];
        foreach (Config::all() as $config) {
            $configs[$config->name] = $config->value;
        }
        return view('admin.settings')->with('configs', $configs);
    }

    public function settings_save(Request $request) {
        $data = $request->all();
        if($request->hasFile('iptv_white_logo')) {
            $file = $request->file('iptv_white_logo');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $website_logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $website_logo);
            $data['iptv_white_logo'] = $website_logo;
        }
        if($request->hasFile('white_logo')) {
            $file = $request->file('white_logo');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $website_logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $website_logo);
            $data['white'] = $website_logo;
        }
        if($request->hasFile('black_logo')) {
            $file = $request->file('black_logo');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $website_logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $website_logo);
            $data['black'] = $website_logo;
        }
        if($request->hasFile('favicon')) {
            $file = $request->file('favicon');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $website_logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $website_logo);
            $data['favicon'] = $website_logo;
        }
        foreach ($data as $key => $value) { $this->UpdateConfig($key, $value); }
        session()->flash('success', '!!');
        return back();
    }

    private function UpdateConfig( $key, $value ) {
        Config::where('name', $key)->update(['value' => $value]);
    }
    /**
     * END SETTINGS
     * START ORIGINS
    */
    public function origins() {
        $origins = Origin::orderBy('created_at', 'desc')->get();
        return view('admin.origins')->with('origins', $origins);
    }
    
    public function origin_save(Request $request) {
        $data = $request->all();
        if($request->hasFile('logo')) {
            $file = $request->file('logo');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $data['logo'] = $poster;
        }else $data['logo'] = '';
        Origin::create($data);
        return redirect()->route('admin.origins')->with('success', 'Origin Inserted Successfully!');
    }

    public function origin_update(Request $request) {
        $data = $request->all();
        unset($data['_token']);
        if($request->hasFile('logo') && $request->logo !== '') {
            $file = $request->file('logo');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $logo = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $logo);
            $data['logo'] = $logo;
        }else unset($data['logo']);
        Origin::whereId($request->id)->update($data);
        return back()->with('success', 'Origin Updated Successfully!');
    }

    public function origin_edit($id) {
        $origin = Origin::whereId($id)->firstOrFail();
        return view('admin.origin_edit')->with('origin', $origin);
    }

    public function origin_delete($id) {
        Origin::whereId($id)->delete();
        return back()->with('success', 'Origin Deleted Successfully!');
    }
    /**
     * END ORIGINS
     * START PRODUCTS
    */
    public function products() {
        $products = Product::orderBy('created_at', 'desc')->get()->map(function($product) {
            $product->orders = $product->orders()->count();
            $product->poster = $product->poster == '' ? asset('noposter.jpeg') : asset('storage/'.$product->poster);
            return $product;
        });
        return view('admin.products')->with('products', $products);
    }

    public function product_insert() {
        $categories = Category::all();
        return view('admin.product_insert')->with(compact('categories'));
    }

    public function product_save(Request $request) {
        $newProductData = $request->all();
        if($request->hasFile('poster')) {
            $file = $request->file('poster');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $newProductData['poster'] = $poster;
        }else {
            $newProductData['poster'] = '';
        }
        $newProductData['slug'] = Str::slug($newProductData['title'], '-');
        Product::create($newProductData);
        return redirect()->route('admin.products')->with('success', 'Product Inserted!');
    }

    public function product_edit($id) {
        $product = Product::findOrFail($id);
        $categories = Category::all();
        return view('admin.product_edit')->with(compact('product', 'categories'));
    }

    public function product_delete($id) {
        Product::find($id)->delete();
        return redirect()->route('admin.products');
    }

    public function product_update(Request $request) {
        $data = $request->all();
        unset($data['_token']);
        if($request->hasFile('poster')) {
            $file = $request->file('poster');
            $fileName = rand(1, 999) . $file->getClientOriginalName();
            $poster = date("Y") . '-' . date("m") . "-" . $fileName;
            $file->storeAs('public', $poster);
            $data['poster'] = $poster;
        }
        Product::find($request->id)->update($data);
        return redirect()->route('admin.product_edit', $request->id)->with('success', 'Product Updated!');
    }

    /**
     * END PRODUCTS
     * START PayPAL
    */

    public function paypal() {
        $is_shuffled = Config::whereName('paypal_shuffled')->first()->value;
        $paypals = Paypal::all();
        return view('admin.paypal')->with(compact('paypals', 'is_shuffled'));
    }

    public function paypal_insert() {
        return view('admin.paypal_insert');
    }

    public function paypal_save(Request $request) {
        Paypal::create( $request->only( 'email', 'client_id', 'max_amount', 'active' ) );
        return redirect()->route('admin.paypal')->with('success', 'PayPal Inserted');
    }

    public function paypal_delete($id) {
        Paypal::whereId($id )->delete();
        return redirect()->route('admin.paypal')->with('success', 'PayPal Removed');
    }

    public function paypal_update(Request $request) {
        $ids = $request->ids;
        $emails = $request->emails;
        $max_amounts = $request->max_amounts;
        $client_ids = $request->client_ids;
        for($i = 0; $i < count($emails); $i ++) {
            Paypal::whereId($ids[$i])->update([
                'email' => $emails[$i],
                'client_id' => $client_ids[$i],
                'max_amount' => $max_amounts[$i]
            ]);
        }
        return back();
    }

    public function paypal_shuffled($current_status) {
        switch($current_status) {
            case 0:
                $new_status = 1;
                break;
            case 1:
                $new_status = 0;
                break;
        }
        Config::whereName('paypal_shuffled')->update([ 'value' =>  $new_status]);
        return back();
    }

    public function paypal_status($id) {
        $current_status = Paypal::whereId($id)->first()->active;
        switch($current_status) {
            case 0:
                $new_status = 1;
                break;
            case 1:
                $new_status = 0;
                break;
        }
        Paypal::whereId($id)->update([ 'active' =>  $new_status]);
        return back();
    }

    /**
     * END PayPAL;
     * END CATEGORIES;
    */

    public function categories() {
        $categories = Category::all();
        return view('admin.categories')->with('categories', $categories);
    }

    public function category_insert() {
        return view('admin.category_insert');
    }

    public function category_save(Request $request) {
        Category::updateOrCreate(['name' => $request->name, 'slug' => $request->slug]);
        return redirect()->route('admin.categories');
    }

    public function category_delete($id) {
        Category::find($id)->delete();
        return redirect()->route('admin.categories');
    }

    public function categories_update(Request $request) {
        $ids = $request->id;
        $names = $request->name;
        $slugs = $request->slug;
        for( $i = 0; $i < count($ids); $i ++ ) {
            Category::find($ids[$i])->update( ['name' => $names[$i], 'slug' => $slugs[$i] ] );
        }
        return redirect()->route('admin.categories');
    }
    /**
     * END CATEGORIES
     * START FAQ
    */
    public function questions() {
        $questions = Question::all();
        return view('admin.questions')->with('questions', $questions);
    }

    public function questions_insert() {
        return view('admin.questions_insert');
    }

    public function questions_edit($id) {
        $question = Question::whereId($id)->first();
        return view('admin.questions_edit')->with('question', $question);
    }

    public function questions_save(Request $request) {
        Question::create([
            'question' => $request->question,
            'answer' => $request->answer,
            'visibility' => 1,
        ]);
        return back()->with('success', true);
    }

    public function questions_update(Request $request) {
        Question::whereId($request->id)->first()->update([
            'question' => $request->question,
            'answer' => $request->answer,
            'visibility' => $request->visibility,
        ]);
        return back()->with('success', true);
    }

    public function questions_delete($id) {
        Question::find($id)->delete();
        return redirect()->route('admin.questions');
    }
    /**
     * END FAQ
     * START LEGAL PAGES
    */
    public function legal_pages() {
        $pages = LegalPage::all();
        return view('admin.legal_pages')->with('pages', $pages);
    }

    public function legal_pages_insert() {
        return view('admin.legal_pages_insert');
    }

    public function legal_pages_save(Request $request) {
        $page = new LegalPage();
        $page->slug = Str::slug($request->title, '-');
        $page->title = $request->title;
        $page->visibility = $request->visibility;
        $page->content = $request->content;
        $page->save();
        return redirect()->route('admin.legal_pages');
    }

    public function legal_pages_delete($id) {
        LegalPage::find($id)->delete();
        return redirect()->route('admin.legal_pages');
    }

    public function legal_pages_edit($id)
    {
        $page = LegalPage::find($id);
        return view('admin.legal_pages_edit')->with('page', $page);
    }

    public function legal_pages_update(Request $request)
    {
        LegalPage::whereId($request->id)->first()->update([
            'title' => $request->title,
            'slug' => $request->slug,
            'visibility' => $request->visibility,
            'content' => $request->content
        ]);
        return back()->with('success', true);
    }

}
