<?php

namespace App\Http\Controllers;

use App\Mail\ContactMail;
use App\Models\Canal;
use App\Models\Category;
use App\Models\Cnumber;
use App\Models\Config;
use App\Models\Country;
use App\Models\Currency;
use App\Models\LegalPage;
use App\Models\Post;
use App\Models\Question;
use App\Models\Server;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class HomeController extends Controller
{
    
    public function getPostData($id) {
        return Post::whereId($id)->first()->content;
    }

    public function search($keyword) {
        return Canal::where('name', 'LIKE', '%'.$keyword.'%')
                    ->where('name', 'NOT LIKE', '%##%')
                    ->with('country')->get()
                    ->map(function($canal) {
                        $canal->category = $canal->country->category->name;
                        $canal->country_name = $canal->country->name;
                        unset($canal->country);
                        return $canal;
                    });
    }

    public function home() {
        $title = Config::whereName('homepage_ctaTitle')->first()->value;
        $description = Config::whereName('seo_description')->first()->value;
        /**
         * STANDARD SERVER;
        */
        $standardServer = Category::whereActive(true)->where('name', 'LIKE', '%standard%')->first();
        if( $standardServer ) {
            $lastStandardCnumber = $standardServer->cnumbers()->whereActive(true)->latest()->first()->number;
            $standardCnumbers = $standardServer->cnumbers()->whereActive(true)->get()->map(function($cnumber) {
                                    $products = $cnumber->products()->whereVisibility(true)->get();
                                    $cnumber->products = $this->addMonthlyPrice($products);
                                    return $cnumber;
                                });
        }else {
            $lastStandardCnumber = 0;
            $standardCnumbers = null;
        }
        /**
         * PREMIUM SERVER;
        */
        $premiumServer = Category::whereActive(true)->where('name', 'LIKE', '%premium%')->first();
        if( $premiumServer ) {
            $lastPremiumCnumber = $premiumServer->cnumbers()->whereActive(true)->latest()->first()->number;
            $premiumCnumbers = $premiumServer->cnumbers()->whereActive(true)->get()->map(function($cnumber) {
                                                    $products = $cnumber->products()->whereVisibility(true)->get();
                                                    $cnumber->products = $this->addMonthlyPrice($products);
                                                    return $cnumber;
                                                });
        }else {
            $lastPremiumCnumber = 0;
            $premiumCnumbers = null;
        }
        /**
         * CONFIGS;
        */
        $currency_id = Config::whereName('currency')->first()->value;
        $currency = Currency::whereId($currency_id)->first();
        $homepage_ctaTitle = Config::whereName('homepage_ctaTitle')->first()->value;
        $homepage_ctaSubtitle = Config::whereName('homepage_ctaSubtitle')->first()->value;
        $homepage_ctaParagraphHtml = Config::whereName('homepage_ctaParagraphHtml')->first()->value;
        $homepage_mainTitle = Config::whereName('homepage_mainTitle')->first()->value;
        $homepage_mainParagraphHtml = Config::whereName('homepage_mainParagraphHtml')->first()->value;
        return view('home')->with(compact(
            'title', 'description',
            'homepage_ctaTitle', 'homepage_ctaSubtitle',
            'homepage_ctaParagraphHtml', 'homepage_mainTitle', 'homepage_mainParagraphHtml',
            'standardCnumbers', 'premiumCnumbers', 'currency',
            'lastStandardCnumber', 'lastPremiumCnumber'
        ));
    }

    private function addMonthlyPrice($products) {
        return $products->map(function($product) {
            if( (int) $product->period !== 1 ) {
                $product->monthly = number_format($product->price / (int) $product->period, 2);
            }
            return $product;
        });
    }

    public function contactus() {
        return view('contactus');
    }

    public function contactus_send(Request $request) {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'subject' => 'required',
            'message' => 'required',
            'h-captcha-response' => 'required'
        ]);
        if ($validator->fails()) {
            return back()->with('error', 'Message Not Sent!')->withErrors($validator)->withInput();
        }else {
            Mail::to(env('MAIL_USERNAME'))->send(new ContactMail($request->message, $request->subject, $request->email));
            return back()->with('success', 'Message Sent Successfully!');
        }
    }
    
    public function category($id)
    {
        $category = Category::findOrFail($id);
        $title = $category->name;
        $products = $category->products()->whereVisibility('1')->get()->map(function($product) {
            $product->poster = $product->poster == '' ? asset('noposter.jpeg') : asset('storage/'.$product->poster);
            return $product;
        });
        return view('category')->with(compact('title', 'products'));
    }


    /**
     * CHANNELS LIST
    */
    public function channels_list() {
        $title = 'Channels List';
        $description = 'find all of our iptv channels so fast by using our search feature, which is not available on most iptv services websites';

        $categories = Category::whereActive(true)->get();

        $cl_defaultCategory = Config::whereName('cl_defaultCategory')->first()->value;
        $defaultCategory = Category::whereId($cl_defaultCategory)->first();

        if( str_contains(strtolower($defaultCategory->name), 'standard') ) {
            $defaultCountry_id = Config::whereName('cl_defaultStandardCountryId')->first()->value;
        }else if( str_contains(strtolower($defaultCategory->name), 'premium') ) {
            $defaultCountry_id = Config::whereName('cl_defaultPremiumCountryId')->first()->value;
        }
        $countries = Category::whereId($defaultCategory->id)->first()->countries;
        $channels = Canal::whereCountryId($defaultCountry_id)->get();
        $defaultCategory_id = $defaultCategory->id;
        return view('channels_list')->with(compact(
            'title', 'description', 'categories', 'countries', 'channels', 'defaultCategory_id', 'defaultCountry_id'
        ));
    }

    public function getChannels($id) {
        return Canal::whereCountryId($id)->get();
    }

    public function changeCategory($cat_id) {
        $category = Category::whereId($cat_id)->firstOrFail();
        if( str_contains(strtolower($category->name), 'standard') ) {
            $defaultCountry_id = Config::whereName('cl_defaultStandardCountryId')->first()->value;
        }else if( str_contains(strtolower($category->name), 'premium') ) {
            $defaultCountry_id = Config::whereName('cl_defaultPremiumCountryId')->first()->value;
        }
        return response()->json([
            'defaultCountry' => $defaultCountry_id,
            'countries' => $category->countries,
            'channels' => Canal::whereCountryId($defaultCountry_id)->get(),
        ]);
    }

    public function faq() {
        $title = 'Frequesntly Asked Questions';
        $description = 'do you have a question about our iptv canada service, try to find the answer now';
        $questions = Question::whereVisibility(1)->get();
        return view('faq')->with( compact('title', 'description', 'questions') );
    }

    public function subscriptions() {
        $title = 'Subscriptions';
        $description = 'Get The Best Subscription Iptv Service offer';
        $currency_id = Config::whereName('currency')->first()->value;
        $currency = Currency::whereId($currency_id)->first();
        /**
         * STANDARD SERVER;
        */
        $standardServer = Category::whereActive(true)->where('name', 'LIKE', '%standard%')->first();
        if( $standardServer ) {
            $lastStandardCnumber = $standardServer->cnumbers()->whereActive(true)->latest()->first()->number;
            $standardCnumbers = $standardServer->cnumbers()->whereActive(true)->get()->map(function($cnumber) {
                                    $products = $cnumber->products()->whereVisibility(true)->get();
                                    $cnumber->products = $this->addMonthlyPrice($products);
                                    return $cnumber;
                                });
        }else {
            $lastStandardCnumber = 0;
            $standardCnumbers = null;
        }
        /**
         * PREMIUM SERVER;
        */
        $premiumServer = Category::whereActive(true)->where('name', 'LIKE', '%premium%')->first();
        if( $premiumServer ) {
            $lastPremiumCnumber = $premiumServer->cnumbers()->whereActive(true)->latest()->first()->number;
            $premiumCnumbers = $premiumServer->cnumbers()->whereActive(true)->get()->map(function($cnumber) {
                                                    $products = $cnumber->products()->whereVisibility(true)->get();
                                                    $cnumber->products = $this->addMonthlyPrice($products);
                                                    return $cnumber;
                                                });
        }else {
            $lastPremiumCnumber = 0;
            $premiumCnumbers = null;
        }
        return view('subscriptions')->with(compact(
            'title', 'description', 'currency',
            'standardCnumbers', 'premiumCnumbers',
            'lastStandardCnumber', 'lastPremiumCnumber'
        ));
    }

    public function post($slug) {
        $post = Post::whereSlug($slug)->first();
        if(!$post) abort(404);
        return view('post')->with('post', $post);
    }

    public function how_to_install(Request $request) {
        $title = 'How To Install';
        $description = 'Your full guide to setup your iptv subscription on your choosed device (s).';
        if( $request->id ) {
            $p = Post::whereId($request->id)->whereVisibility(true)->firstOrFail();
        }else $p = null;
        $posts = Post::whereVisibility(true)->orderBy('created_at', 'desc')->select([ 'id', 'title', 'icon' ])->get();
        return view('how_to_install')->with(compact('title', 'description', 'posts', 'p'));
    }

    public function page($slug) {
        $page = LegalPage::whereSlug($slug)->first();
        if(!$page) abort(404);
        return view('post')->with('post', $page);
    }
}
